*****************************************************************
*
*  Sample Printer Driver
*
*  This sample driver does simple communication with a port driver
*  and saves a QD Pict of the pages to be imaged to the
*  */SYSTEM/DRIVERS folder.
*
*  By Matt Deatherage
*
*  Copyright 1990, Apple Computer, Inc.  All rights reserved.
*
*
*  Revision history:
*
*  7/15/90   Version 1.0
*
*  First take.  All routines present and functional.
*
*  8/07/90   Version 1.1
*
*  Code rearranged and a MAKEFILE added.  PrDefault and PrValidate rewritten
*  to behave themselves instead of tinkling in the print record.
*
*  8/18/90   Version 1.2
*
*  The Dave version.  Incorporated his comments.
*
*****************************************************************

             mcopy picter.macros
             copy  drvrequates.asm      ; all of our equates
             copy  2/ainclude/e16.quickdraw
             copy  2/ainclude/e16.window
             copy  2/ainclude/e16.print

**********************************************************************
*
* PictDriver main entry point
*
*  Inputs:   Printer driver entry - precomputed X register
*
* Outputs:   None - branches to EndOurWorld in each routine
*
**********************************************************************

Picter       START

;
;  First, the standard new printer driver header
;

             dc    i2'0'                ; new style driver
             dc    i2'(ListEnd-PrDriverList)/4'

             jsr   MakeOurWorld

;
;  After setting things up, we have the following information in the
;  registers:
;
;  X = precomputed jump table offset to the routine to call
;  A = any error code from establishing our world (memory errors, etc.)
;  carry flag = set if we had to create our DP.
;
;  If the carry is set and A is non-zero, there was an error and our world
;  does not exist.  We can't handle this here generically because each
;  routine has different parameters to pull off the stack.  Instead, we
;  pass this register information to each of the routines for proper error
;  handling.  For example, if we store stuff between calls on our direct page,
;  it's an error for PrOpenPage to get the carry set.  That would mean that
;  the driver was unloaded after PrOpenDoc, and this isn't good.
;

             jmp   (PrDriverList,x)     ; go do the routine
;                                         (all routines have to be in this seg)


PrDriverList dc    i4'PrDefault'        ; $0913
             dc    i4'PrValidate'       ; $0A13
             dc    i4'PrStlDialog'      ; $0B13
             dc    i4'PrJobDialog'      ; $0C13
             dc    i4'PrDriverVer'      ; $2313
             dc    i4'PrOpenDoc'        ; $0E13
             dc    i4'PrCloseDoc'       ; $0F13
             dc    i4'PrOpenPage'       ; $1013
             dc    i4'PrClosePage'      ; $1113
             dc    i4'PrPicFile'        ; $1213
             dc    i4'PrReserved'       ; $1313 is reserved, so get out
             dc    i4'PrError'          ; $1413
             dc    i4'PrSetError'       ; $1513
             dc    i4'GetDeviceName'    ; $1713
             dc    i4'PrPixelMap'       ; $0D13
             dc    i4'PrGetPrinterSpecs' ; $1813
             dc    i4'PrGetPgOrientation' ;$3813
ListEnd      anop

             END

**********************************************************************
*
* PrDefault fills the given handle with this driver's default print
* record, which depends on the current screen resolution.
*
*  Inputs:   Print record handle on the stack
*
* Outputs:   None
*
**********************************************************************

PrDefault    START
             USING PicterData

             ldy   #4
             jsr   CheckTheWorld

; Parameters start at ParamStart,s (see MakeOurWorld)

             lda   ParamStart,s
             sta   <PrRecHndl
             lda   ParamStart+2,s
             sta   <PrRecHndl+2

             pei   <PrRecHndl+2         ; doing this separately avoids changing
             pei   <PrRecHndl           ; the stack offsets - either way's fine
             _CheckHandle
             bcc   HandleOK
             lda   #noPrintRecord       ; change it to a print manager error
             brl   DefOut               ; get out of PrDefault

HandleOK     pushlong #DfPRecBase       ; default print record base addr
             pei   <PrRecHndl+2
             pei   <PrRecHndl
             pushlong #PrRecSize
             _PtrToHand                 ; fill in the print record

             jsr   CalcMetrics          ; fix up the metrics

             lda   #0                   ; no error

DefOut       tax
             ldy   #4
             brl   EndOurWorld


             END

**********************************************************************
*
* PrValidate looks at those values which could be invalid for our
* printer and if they are invalid, corrects them.
*
*  Inputs:   Word space for boolean result
*            Print record handle on the stack
*
* Outputs:   recAdjusted flag
*
**********************************************************************

PrValidate   START
             USING PicterData

             ldy   #4
             jsr   CheckTheWorld

; Parameters start at ParamStart,s (see MakeOurWorld)

             lda   ParamStart,s
             sta   <PrRecHndl
             lda   ParamStart+2,s
             sta   <PrRecHndl+2

             pei   <PrRecHndl+2         ; doing this separately avoids changing
             pei   <PrRecHndl           ; the stack offsets - either way's fine
             _CheckHandle
             bcc   OK0
             ldx   #noPrintRecord       ; change it to a print manager error
             ldy   #4                   ; number of bytes to remove
             brl   ValOut2              ; get out of PrValidate

;
; We use the same value used by CalcMetrics - RectChanged - to indicate
; if the print record has changed any while validating.
;

OK0          stz   <RectChanged

             ldy   #4
             lda   [PrRecHndl],y
             sta   <temp                ; save it on direct page
             ora   #$8000               ; lock it
             sta   [PrRecHndl],y
             dey
             dey
             lda   [PrRecHndl],y
             sta   <PrRecPtr+2
             lda   [PrRecHndl]
             sta   <PrRecPtr            ; and dereference it

             ldy   #oprInfo+oiDev       ; the iDev parameter
             lda   [PrRecPtr],y         ; get the iDev field
             cmp   #OuriDev             ; is it ours?
             beq   OK1
             brl   doDefault

OK1          ldy   #oprInfo+oiVRes
             lda   [PrRecPtr],y
             cmp   #ouriVRes            ; is it ours?
             beq   OK2
             lda   #ouriVRes
             sta   [PrRecPtr],y
             inc   <RectChanged         ; note we did something

OK2          pha
             _GetMasterSCB
             pla
             sta   <MasterSCB
             and   #$0080
             beq   Test320

Test640      ldy   #oprInfo+oiHRes
             lda   [PrRecPtr],y
             cmp   #ouriHRes640
             beq   OK3
             lda   #ourIHRes640
             sta   [PrRecPtr],y
             inc   <RectChanged
OK3          bra   doneHRes

Test320      ldy   #oprInfo+oiHRes
             lda   [PrRecPtr],y
             cmp   #ouriHRes320
             beq   doneHRes
             lda   #ouriHRes320
             sta   [PrRecPtr],y
             inc   <RectChanged

doneHRes     jsr   CalcMetrics
             lda   <RectChanged
             cmp   #$FFFF
             beq   DoDefault

             ldy   #oprStl+owDev
             lda   [PrRecPtr],y
             and   #$FFFF-AppleBitMask  ; don't care about Apple's bits
             cmp   #OurwDev
             beq   CheckMethod
             lda   [PrRecPtr],y
             and   #AppleBitMask        ; leave Apple's reserved bits set
             ora   #OurwDev
             sta   [PrRecPtr],y
             inc   <RectChanged


CheckMethod  ldy   #oprJob+objDocLoop
             lda   [PrRecPtr],y
             and   #$00FF               ; only one byte, please
             cmp   #OurBJDocLoop        ; should be spool-only
             beq   ZeroIdleProc
             lda   [PrRecPtr],y
             and   #$FF00               ; clear out low byte
             ora   #OurBJDocLoop        ; or in the low byte
             sta   [PrRecPtr],y
             inc   <RectChanged

ZeroIdleProc ldy   #oprJob+opIdleProc   ; always zero this out
             lda   #0
             sta   [PrRecPtr],y
             iny
             iny
             sta   [PrRecPtr],y

             lda   <RectChanged         ; this is zero if the rects were OK
             beq   ItsZero
             lda   #1
ItsZero      sta   ParamStart+4,s       ; so use it as changed flag
             ldx   #0

ValOut       lda   <Temp
             ldy   #4                   ; remove four bytes of parameters
             sta   [<PrRecHndl],y       ; and it's an index, too!  At no cost!
ValOut2      brl   EndOurWorld

doDefault    pei   <PrRecHndl+2
             pei   <PrRecHndl
             _PrDefault
             tax                        ; save the error code
             lda   #TRUE
             sta   ParamStart+4,s       ; set the changed flag

             bra   ValOut

             END

**********************************************************************
*
* PrStlDialog conducts the "page setup" dialog.
*
*  Inputs:   Word space for confirmation flag
*            Print record handle
*
* Outputs:   confirmation flag
*
**********************************************************************

PrStlDialog  START
             USING PicterData

             ldy   #4
             jsr   CheckTheWorld

             lda   ParamStart,s
             sta   <PrRecHndl
             lda   ParamStart+2,s
             sta   <PrRecHndl+2

             ldy   #4
             lda   [PrRecHndl],y
             sta   <temp                ; save old handle attributes
             ora   #$8000               ; lock the print record handle
             sta   [PrRecHndl],y
             dey
             dey
             lda   [PrRecHndl],y
             sta   <PrRecPtr+2
             lda   [PrRecHndl]
             sta   <PrRecPtr            ; and dereference it

             jsr   ConductStyleDialog
             pha                        ; save for a moment

             lda   <temp
             ldy   #4
             sta   [<PrRecHndl],y       ; restore the handle attributes

             pla                        ; restore the dialog result
             sta   ParamStart+4,s       ; space for result

             ldy   #4
             ldx   #0
             brl   EndOurWorld

             END

**********************************************************************
*
* PrJobDialog conducts the "print" dialog.
*
*  Inputs:   Word space for confirmation flag
*            Print record handle
*
* Outputs:   confirmation flag
*
**********************************************************************

PrJobDialog  START
             USING PicterData

             ldy   #4
             jsr   CheckTheWorld

             lda   ParamStart,s
             sta   <PrRecHndl
             lda   ParamStart+2,s
             sta   <PrRecHndl+2

             ldy   #4
             lda   [PrRecHndl],y
             sta   <temp                ; save handle attributes
             ora   #$8000               ; lock the print record handle
             sta   [PrRecHndl],y
             dey
             dey
             lda   [PrRecHndl],y
             sta   <PrRecPtr+2
             lda   [PrRecHndl]
             sta   <PrRecPtr            ; and dereference it

             jsr   ConductJobDialog
             pha                        ; save for a moment

             lda   <temp
             ldy   #4
             sta   [<PrRecHndl],y       ; restore the handle attributes

             pla                        ; restore the dialog result
             sta   ParamStart+4,s       ; space for result

             ldy   #4
             ldx   #0
             brl   EndOurWorld

             END

**********************************************************************
*
* PrPixelMap prints all or part of the specified pixel map.  For our
* picter driver, we create a picture with the pixel map in it and
* write it to the disk.
*
*  Inputs:   srcLocPtr, srcRectPtr, colorFlag (ignored)
*
* Outputs:   None
*
**********************************************************************

PrPixelMap   START
             USING PicterData

             ldy   #10                  ; 10 bytes of parameters
             jsr   CheckTheWorld

;
;  We don't use a port driver here, but normally we would make sure it's
;  safe to print at this point.
;
;            pha                        ; space for result
;            _PrDevIsItSafe             ; is the port around?
;            pla                        ; the result
;            bne   OKToPrint            ; the port is safe
;            lda   #PortNotOn           ; error $1302
;            sta   <OurError
;            brl   PixMapOut            ; get out of here
;

OKToPrint    pha
             pha                        ; space for result
             _GetPort
             pulllong <OldPort          ; save it

             pha
             pha
             _GetCursorAdr
             pulllong <OldCursor        ; save this too

             _WaitCursor                ; watch time

             stz   <OurError            ; clear this out

;
;  We don't create a new print record handle because we don't have to
;  use it to communicate with the caller and we don't need it in this case.
;  In many cases it will be easier for PrPixelMap to allocate a print
;  record and call the deferred printing code.  Do whichever suits your
;  needs best.
;

             pha
             pha                        ; space for result
             pushlong #GrafPortSize
             pei   <MyID
             pea   $C000                ; locked and fixed
             pushlong #0
             _NewHandle
             tax                        ; save the error code
             pulllong <PortHandle       ; save the handle
             txa                        ; restore the error code
             bcs   PixMapOut

             lda   [<PortHandle]
             sta   <PortPointer
             ldy   #2
             lda   [<PortHandle],y
             sta   <PortPointer+2

             pei   <PortPointer+2
             pei   <PortPointer
             _OpenPort                  ; set it up as a grafPort

DoPicture    clc
             lda   <PortPointer
             adc   #oboundsRect
             sta   <RectPointer
             lda   <PortPointer+2
             adc   #0
             sta   <RectPointer+2

             jsr   OpenPICTFile         ; create the new picture and file
             bcs   PixMapOut2

;
;  Our parameters start at ParamStart,s and will change as we push duplicate
;  copies of them on the stack, so the offsets stay the same.
;

             lda   ParamStart+8,s       ; high word of srcLocPtr
             pha
             lda   ParamStart+8,s       ; low word of srcLocPtr
             pha
             lda   ParamStart+8,s       ; high word of srcRectPtr
             pha
             lda   ParamStart+8,s       ; low word of srcRectPtr
             pha
             lda   #0
             pha                        ; startX
             pha                        ; startY
             pea   modeCopy
             _PPToPort                  ; it's now in the picture

             jsr   ClosePICTFile        ; write it and close it
             bcs   PixMapOut2
             lda   #0                   ; no error

;
;  Note that if we were really printing, we'd want to form feed at the
;  end of the pixel map - one job per page, please
;

;
;  PixMapOut2 gets out of here after taking care of all the stuff we
;  changed around and allocated (cursors, grafPorts, etc.)
;

PixMapOut2   pha                        ; save error code on stack

             pei   <OldPort+2
             pei   <OldPort
             _SetPort

             pei   <PortPointer+2
             pei   <PortPointer
             _ClosePort                  ; always close what you open

             pei   <PortHandle+2
             pei   <PortHandle
             _DisposeHandle

             pei   <OldCursor+2
             pei   <OldCursor
             _SetCursor

             pla                        ; put error code back in A

PixMapOut    tax
             ldy   #10                  ; 10 bytes to pull
             brl   EndOurWorld

             END

**********************************************************************
*
* PrOpenDoc prepares us for the print loop.  We set up some parameters
* for the loop and get ready for a grand old time.
*
*  Inputs:   Space for returned GrafPort
*            Pointer to printing GrafPort (or NIL to allocate one)
*            Print record handle
*
* Outputs:   Pointer to the printing GrafPort
*
**********************************************************************

PrOpenDoc    START
             USING PicterData

             ldy   #8
             jsr   CheckTheWorld

;
;  We don't use a port driver here, but normally we would make sure it's
;  safe to print at this point.
;
;            pha                        ; space for result
;            _PrDevIsItSafe             ; is the port around?
;            pla                        ; the result
;            bne   OKToPrint            ; the port is safe
;            lda   #PortNotOn           ; error $1302
;            sta   <OurError
;            brl   OpenDocOut           ; get out of here
;

OKToPrint    pha
             pha                        ; space for result
             _GetPort
             pulllong <OldPort          ; save the current port

             pha
             pha
             _GetCursorAdr
             pulllong <OldCursor        ; save the old cursor

             _WaitCursor                ; note this may take a while

             stz   <OurError            ; clear out the internal error variable

             pha                        ; space for result
             lda   ParamStart+8,s       ; high word of print record handle
             pha
             lda   ParamStart+8,s       ; low word
             pha
             _PrValidate                ; check out the print record
             pla                        ; like I care if it changed...

             jsr   StartStatusMessage   ; initialize our little box of fun

             lda   #PrOpenDocMsg
             sta   <ParamLongA
             lda   #PrOpenDocMsg|-16
             sta   <ParamLongA+2
             jsr   StatusMessage        ; display our OpenDoc message

             lda   ParamStart,s
             ora   ParamStart+2,s       ; is this NIL?
             beq   GetGrafPort          ; yes, allocate a graf port
             lda   ParamStart,s
             sta   <PortPointer
             lda   ParamStart+2,s
             sta   <PortPointer+2       ; put it on direct page
             stz   <WeGotPort           ; initialize this to FALSE
             bra   GotGrafPort

GetGrafPort  pha
             pha                        ; space for result
             pushlong #GrafPortSize
             pei   <MyID
             pushword #$C000            ; locked and fixed
             lda   #0
             pha
             pha                        ; no location
             _NewHandle
             tax                        ; save error code
             pulllong <PortHandle
             stx   <OurError
             bcs   OpenDocOut           ; no printing for us

             ldy   #2
             lda   [PortHandle],y
             sta   <PortPointer+2
             lda   [PortHandle]
             sta   <PortPointer

             pei   <PortPointer+2
             pei   <PortPointer
             _OpenPort

             lda   #1
             sta   <WeGotPort

GotGrafPort  anop

             stz   <PageNumber          ; clear this out

             lda   <PortPointer
             sta   ParamStart+8,s       ; the return parameter
             lda   <PortPointer+2
             sta   ParamStart+10,s

OpenDocOut   ldx   <OurError
             ldy   #8
             brl   EndOurWorld

             END

**********************************************************************
*
* PrCloseDoc undoes everything done by PrOpenDoc except the cursor
* and restoring the current port.
*
*  Inputs:   The printing grafPort pointer
*
* Outputs:   None.
*
**********************************************************************

PrCloseDoc   START

;
;  In this routine, it is very important that the direct page have
;  already been set up.  If it got set up on the entrance to this call,
;  some idiot unloaded the printer driver in the middle of the print
;  loop and all our stuff is invalid.
;

             bcc   Whew                 ; nothing's wrong yet
             ldy   #4                   ; 4 bytes of parameters
             ldx   #PrBozo              ; PrBozo error
             brl   EndOurWorld

Whew         lda   <OurError
             cmp   #PortNotOn           ; is this the error
             bne   Whew2                ; still good
             ldy   #4
             ldx   #0                   ; don't return error, just return
             brl   EndOurWorld

Whew2        lda   ParamStart,s
             sta   <PortPointer
             lda   ParamStart+2,s
             sta   <PortPointer+2

             pei   <OldPort+2
             pei   <OldPort
             _SetPort                   ; the previous port is now restored

             pei   <PortPointer+2
             pei   <PortPointer
             _ClosePort

             lda   <WeGotPort           ; did we allocate this?
             beq   CloseDialog

             pei   <PortHandle+2
             pei   <PortHandle
             _DisposeHandle

CloseDialog  jsr   FinishStatusMessage

CloseDocOut  ldx   #0
             ldy   #4
             brl   EndOurWorld

             END

**********************************************************************
*
* PrOpenPage captures, somehow, an image of a page to be printed if
* and only if the page falls within the range of pages to be printed
* as specified in the print record.  We do this with QuickDraw pictures,
* an excellent choice since they're what we're writing to disk anyway.
*
*  Inputs:   Pointer to the printing grafPort
*            Pointer to a pageFrame rectangle
*
* Outputs:   None
*
**********************************************************************

PrOpenPage   START
             USING PicterData
;
;  In this routine, it is very important that the direct page have
;  already been set up.  If it got set up on the entrance to this call,
;  some idiot unloaded the printer driver in the middle of the print
;  loop and all our stuff is invalid.
;

             bcc   Whew                 ; nothing's wrong yet
             ldy   #8                   ; 8 bytes of parameters
             ldx   #PrBozo              ; PrBozo error
             brl   EndOurWorld

Whew         lda   <OurError            ; return if any error is here
             beq   Whew2                ; still good
             ldy   #8
             ldx   #0                   ; don't return error, just return
             brl   EndOurWorld

Whew2        inc   <PageNumber          ; we're on the next page now

             ldy   #4
             lda   [PrRecHndl],y
             sta   <Temp2               ; save handle attributes
             ora   #$8000               ; lock the print record handle
             sta   [PrRecHndl],y
             dey
             dey
             lda   [PrRecHndl],y
             sta   <PrRecPtr+2
             lda   [PrRecHndl]
             sta   <PrRecPtr            ; and dereference it

             lda   <PageNumber
             ldy   #oprJob+oiFstPage
             cmp   [<PrRecPtr],y
             bge   Whew3                ; Dave would rather see BCS - tough
             lda   #0                   ; no error to be out of page range
             brl   OpenPageOut          ; sorry, no printing
Whew3        ldy   #oprJob+oiLstPage
             dec   a                    ; make it one smaller
             cmp   [<PrRecPtr],y
             blt   Whew4                ; Dave would rather see BCC - tough
             lda   #0                   ; no error to be out of page range
             brl   OpenPageOut          ; don't print

Whew4        lda   #PrOpenPageMsg
             sta   <ParamLongA
             lda   #PrOpenPageMsg|-16
             sta   <ParamLongA+2
             jsr   StatusMessage        ; PrOpenPage message

             lda   ParamStart+4,s
             sta   <PortPointer
             lda   ParamStart+6,s
             sta   <PortPointer+2

             pei   <PortPointer+2
             pei   <PortPointer
             _SetPort                   ; it's now current

             pei   <PortPointer+2
             pei   <PortPointer
             _InitPort                  ; reset it to the default

             pushlong #LocInfoSpace     ; space for LocInfo games
             _GetPortLoc

             stz   LocInfoSpace+8
             stz   LocInfoSpace+10
             stz   LocInfoSpace+12      ; zero out the boundsRect so
             stz   LocInfoSpace+14      ; no drawing occurs

             pushlong #LocInfoSpace
             _SetPortLoc

             lda   ParamStart,s
             ora   ParamStart+2,s       ; is the frameRect NIL?
             beq   getrPage             ; if so, get rPage

             lda   ParamStart,s
             sta   <RectPointer
             lda   ParamStart+2,s
             sta   <RectPointer+2       ; rectangle now set up for OpenPICTFile
             bra   gotTheRect

getrPage     lda   #oprInfo+orPage
             clc
             adc   <PrRecPtr
             sta   <RectPointer
             lda   #0
             adc   <PrRecPtr+2
             sta   <RectPointer+2

gotTheRect   jsr   OpenPICTFile
             bcs   OpenPageOut          ; error = no printing

;
;  Now set the clipRgn and visRgn to a rectangular region the size of the
;  framing rectangle
;

             pha
             pha
             _NewRgn                    ; we got the region
             pulllong <Temp             ; save it for a moment

             pei   <Temp+2
             pei   <Temp                ; the region handle
             pei   <RectPointer+2
             pei   <RectPointer         ; the framing rectangle
             _RectRgn                   ; now it's that size

             pei   <Temp+2
             pei   <Temp
             _SetClip                   ; copies this into the clipRgn

             pei   <Temp+2
             pei   <Temp
             _SetVisRgn                 ; copy it into the visRgn as well

             pei   <Temp+2
             pei   <Temp
             _DisposeRgn                ; throw away what we allocate

;
;  That's basically it.  Now the application will draw into the printing
;  grafPort, which will go into our picture and we write it when we close
;  the page.  Nifty.
;

OpenPageOut  tax                        ; put the error code in X
             lda   <temp2
             ldy   #4
             sta   [<PrRecHndl],y       ; restore the handle attributes

             ldy   #8
             brl   EndOurWorld          ; and return to the real world

             END

**********************************************************************
*
* PrClosePage closes out the page.  For us, we write it out to disk
* here.
*
*  Inputs:   Our stuff on direct page and a printing GrafPort
*
* Outputs:   None
*
**********************************************************************

PrClosePage  START
             USING PicterData

;
;  In this routine, it is very important that the direct page have
;  already been set up.  If it got set up on the entrance to this call,
;  some idiot unloaded the printer driver in the middle of the print
;  loop and all our stuff is invalid.
;

             bcc   Whew                 ; nothing's wrong yet
             ldy   #4                   ; 4 bytes of parameters
             ldx   #PrBozo              ; PrBozo error
             brl   EndOurWorld

Whew         lda   <OurError
             cmp   #PortNotOn           ; is this the error
             bne   Whew2                ; still good
             ldy   #4
             ldx   #0                   ; don't return error, just return
             brl   EndOurWorld

Whew2        anop

;
;  Note that if we did support immediate mode, we'd want to do a form feed
;  here.
;

             lda   #PrClosePageMsg
             sta   <ParamLongA
             lda   #PrClosePageMsg|-16
             sta   <ParamLongA+2
             jsr   StatusMessage        ; PrClosePage message

             jsr   ClosePICTFile        ; close out our printing

ClosePageOut tax
             ldy   #4
             brl   EndOurWorld          ; that's all!

             END

**********************************************************************
*
* For any real printer driver, PrPicFile is the main course of the
* printing meal.  This routine takes all the saved up images, spooled
* either to memory or to disk and prints them.  This routine will be
* pretty big for most drivers.
*
* Aha, but this isn't most drivers.  This is pretty small and does no
* actual imaging.  See the article for more on PrPicFile.
*
*  Inputs:   PrintRecordHandle, printing GrafPort and statusRec
*
* Outputs:   None
*
**********************************************************************

PrPicFile    START
             USING PicterData

;
;  In this routine, it is very important that the direct page have
;  already been set up.  If it got set up on the entrance to this call,
;  some idiot unloaded the printer driver in the middle of the print
;  loop and all our stuff is invalid.
;

             bcc   Whew                 ; nothing's wrong yet
             ldy   #12                  ; 12 bytes of parameters
             ldx   #PrBozo              ; PrBozo error
             brl   EndOurWorld

Whew         lda   <OurError
             cmp   #PortNotOn           ; is this the error
             bne   Whew2                ; still good
             ldy   #12
             ldx   #0                   ; don't return error, just return
             brl   EndOurWorld

Whew2        anop

             jsr   StartStatusMessage

             lda   #PrPicFileMsg
             sta   <ParamLongA
             lda   #PrPicFileMsg|-16
             sta   <ParamLongA+2
             jsr   StatusMessage        ; "PrPicFile"


;
;  Even though we're not printing, we'll go through some of the motions
;  to make the system happy.  For example, we'll set up the statusRec and
;  call the idleProc procedure at least once just to make it happy (and to
;  demonstrate this stuff).
;

             lda   ParamStart+4,s
             ora   ParamStart+6,s       ; is this NIL?
             beq   GetGrafPort          ; yes, allocate a graf port
             lda   ParamStart+4,s
             sta   <PortPointer
             lda   ParamStart+6,s
             sta   <portPointer+2       ; put it on direct page
             stz   <WeGotPort           ; initialize this to FALSE
             bra   GotGrafPort

GetGrafPort  pha
             pha                        ; space for result
             pushlong #GrafPortSize
             pei   <MyID
             pushword #$C000            ; locked
             lda   #0
             pha
             pha                        ; no location
             _NewHandle
             tax                        ; save error code
             pulllong <PortHandle
             stx   <OurError
             bcc   NoErrorHere
             txa                        ; put our error in A for PicFileOut
             brl   PicFileOut           ; no printing for us

NoErrorHere  ldy   #2
             lda   [PortHandle],y
             sta   <PortPointer+2
             lda   [PortHandle]
             sta   <PortPointer

             lda   #1
             sta   <WeGotPort

GotGrafPort  anop

             pei   <PortPointer+2
             pei   <PortPointer
             _OpenPort                  ; make it nice and grafPort-ical

             lda   ParamStart,s         ; checking the statusRecPtr
             ora   ParamStart+2,s       ; is it zero
             beq   NewStatusRec         ; yes, get a new one
             lda   ParamStart,s
             sta   <StatusRecPtr
             lda   ParamStart+2,s
             sta   <StatusRecPtr+2
             stz   <StatusRecHandle
             stz   <StatusRecHandle+2   ; know that we didn't use this
             bra   GotStatusRec

NewStatusRec pha
             pha                        ; space for result
             pushlong #StatusRecSize    ; size
             pei   <MyID                ; owner ID
             pushword #$C000            ; locked and fixed
             lda   #0
             pha
             pha                        ; no location
             _NewHandle
             tax
             pulllong <StatusRecHandle
             txa
             bcc   DerefSRHandle
             brl   PicFileOut           ; get out of here

DerefSRHandle lda   [<StatusRecHandle]
             sta   <StatusRecPtr
             ldy   #2
             lda   [<StatusRecHandle],y
             sta   <StatusRecPtr+2

             pushlong #MyStatusRec      ; our default copy
             pei   <StatusRecPtr+2
             pei   <StatusRecPtr        ; our destination
             pushlong #StatusRecSize
             _BlockMove                 ; move it there

GotStatusRec ldy   #ohPrint
             lda   <PrRecHndl
             sta   [<StatusRecPtr],y
             iny
             iny
             lda   <PrRecHndl+2
             sta   [<StatusRecPtr],y

             iny
             iny
             lda   <PortPointer
             sta   [<StatusRecPtr],y
             iny
             iny
             lda   <PortPointer+2
             sta   [<StatusRecPtr],y

             ldy   #2
             lda   [PrRecHndl],y
             sta   <PrRecPtr+2
             lda   [PrRecHndl]
             sta   <PrRecPtr            ; dereference the print record handle

             jsr   doIdleProc           ; make it happy

;
;  Here's where the real stuff goes - we just don't have any
;

             pei   <PortPointer+2
             pei   <PortPointer
             _ClosePort

             lda   <WeGotPort
             beq   PortGone

             pei   <PortHandle+2
             pei   <PortHandle
             _DisposeHandle

PortGone     lda   <StatusRecHandle
             ora   <StatusRecHandle+2
             beq   StatusRecGone

             pei   <StatusRecHandle+2
             pei   <StatusRecHandle
             _DisposeHandle

StatusRecGone anop

             jsr   FinishStatusMessage

             pei   <OldCursor+2
             pei   <OldCursor
             _SetCursor
             bcs   PicFileOut

             lda   #0                   ; no errors!  yay!

PicFileOut   tax                        ; error in X
             ldy   #12                  ; twelve bytes of parameters
             brl   EndOurWorld

             END

**********************************************************************
*
* PrError returns the last driver error we have.
*
*  Inputs:   Space for result
*
* Outputs:   Error word on stack
*
**********************************************************************

PrError      START

             ldy   #0
             jsr   CheckTheWorld

             lda   <OurError            ; and here it is
             sta   ParamStart,s         ; and there it goes
             ldy   #0
             ldx   #$FFFF
             brl   EndOurWorld

             END

**********************************************************************
*
*  PrSetError sets our internal error variable to the parameter.
*
*  Inputs:   Error word on the stack
*
* Outputs:   None
*
**********************************************************************

PrSetError   START

             ldy   #2
             jsr   CheckTheWorld

             lda   ParamStart,s         ; get the error
             sta   <OurError
             ldx   #$FFFF               ; don't change the error status
             ldy   #2
             brl   EndOurWorld

             END

**********************************************************************
*
* GetDeviceName (also known as PrChanged) is called when drivers are
* changed.  Although we don't have anything to do here, we'll give
* a sample of what we would do if we could run on a network.
*
*  Inputs:   None
*
* Outputs:   None
*
**********************************************************************

GetDeviceName START

             ldy   #0
             jsr   CheckTheWorld

             pushlong #DeviceNameStr
             ldx   #$1913
             jsl   $E10000              ; _PrDevPrChanged

             ldy   #0                   ; no bytes to return

             ldx   #0                   ; no error
             brl   EndOurWorld

DeviceNameStr str  '='                  ; in AppleTalk NBP format
             str   'Picter'
             str   '*'

             END

**********************************************************************
*
* PrDriverVer returns the version word for this driver.
*
*  Inputs:   Space for result
*
* Outputs:   Version of this driver
*
**********************************************************************

PrDriverVer  START

             ldy   #0
             jsr   CheckTheWorld

             lda   #OurVersion
             sta   ParamStart,s         ; the space on the stack
             ldy   #0                   ; no bytes to remove
             ldx   #0                   ; no errors
             brl   EndOurWorld

             END

**********************************************************************
*
* PrGetPrinterSpecs returns characteristics of the printer.
*
*  Inputs:   Two word spaces
*
* Outputs:   Printer type and characteristics
*
**********************************************************************

PrGetPrinterSpecs  START

             ldy   #0
             jsr   CheckTheWorld

             lda   #OuriDev
             sta   ParamStart+2,s

             lda   #OurCharcs
             sta   ParamStart,s

             ldy   #0                   ; no bytes to remove
             ldx   #0                   ; no errors
             brl   EndOurWorld

             END

**********************************************************************
*
* PrGetPgOrientation returns the page orientation from a print
* record for our driver.
*
*  Inputs:   Word Space for result
*            Print record handle
*
* Outputs:   Page orientation (0 = portrait, 1 = landscape)
*
**********************************************************************

PrGetPgOrientation  START

             ldy   #4
             jsr   CheckTheWorld

             lda   ParamStart,s
             sta   <Temp
             lda   ParamStart+2,s
             sta   <Temp+2              ; temp = print record handle

             lda   [<Temp]
             sta   <Temp2
             ldy   #2
             lda   [<Temp],y
             sta   <Temp2+2             ; temp2 = print record pointer

;
;  We do this into Temp in case some bozo calls this in the middle of the
;  printing loop, where our print record fields are filled
;

             ldy   #oprStl+owDev
             lda   [<Temp2],y           ; the wDev field
             and   #$0002               ; clear all but bit 2
             bne   DoLandscape

DoPortrait   lda   #Portrait
             bra   OrientOut

DoLandscape  lda   #Landscape

OrientOut    sta   ParamStart+4,s
             ldx   #0                   ; no errors
             ldy   #4                   ; four bytes to remove
             brl   EndOurWorld

             END


**********************************************************************
*
* PrReserved (formerly PrControl) is treated as if there are no
* parameters, even though PrControl had some, so don't even THINK
* about making this call.
*
*  Inputs:   None
*
* Outputs:   None
*
**********************************************************************

PrReserved   START

             ldx   #$0002               ; as in Tool Locator function not found
             ldy   #0
             brl   EndOurWorld

             END

**********************************************************************
*
*  MakeOurWorld sets up a private direct page and sets the data bank
*  register to our code bank.  We check our internal storage to see
*  if our direct page already exists, and if not, we create it.
*
*  Inputs:   None.  Assumes no bytes between the two RTL addresses and
*            the RTS address for this routine.
*
* Outputs:   Direct page and data bank registers on stack.  The
*            carry is set if our DP did not exist on entry.  Preserves
*            the X register.
*
**********************************************************************

;
;  When this routine is finished, the stack looks like this:
;
;  | Previous contents |
;  +-------------------+
;  | Parameters (10,s) |
;  +-------------------+
;  | RTL2        (7,s) |
;  +-------------------+
;  | RTL1        (4,s) |
;  +-------------------+
;  | Old DP reg  (2,s) |
;  +-------------------+
;  | Old DBR     (1,s) |
;  +-------------------+
;  |                   |  <-- SP
;
;  We've already equated ParamStart to 10 for this reason.  If you want to
;  allocate DP space on the stack, change ParamStart accordingly.
;
;  NOTE:  THIS ROUTINE MUST SUPPORT DRIVER REENTRANCY!  When we make any
;  Port Driver or other Tool Locator calls, someone could call one of our
;  routines in the middle of one of our routines (GetDeviceName is the
;  most likely candidate).  This MUST be reentrant.
;

MakeOurWorld START
             USING PicterData

             pla                        ; the RTS address
             phd                        ; save the old DP
             phb                        ; save the old DBR
             pha                        ; put the RTS address back

             phk
             plb                        ; now we can use 16-bit addressing

;
;  We look at our internal variable OurDP to see if it's NIL.  We assemble
;  it to zero so it will be that way when it's loaded.  When we allocate our
;  direct page, we change it to that value.  When the Print Manager unloads
;  our driver, our DP will go away because we allocate it with our user ID.
;  When it gets reloaded, OurDP will be zero again.
;  Therefore, if OurDP is zero we don't have a direct page yet.
;
             lda   OurDP
             bne   GotDP                ; we've got it

NoDP         phx                        ; save the X register
             pha                        ; space for result
             _MMStartUp                 ; leave our ID on the stack
             lda   1,s                  ; but get a copy

             pha
             pha                        ; space for result

             pea   $0000
             pea   $0100                ; pushlong #$100 without destroying A
             pha                        ; our user ID
             pea   $C015                ; locked, fixed, no bank cross, page
*                                       ; aligned and fixed bank
             lda   #0
             pha
             pha                        ; location in bank zero
             _NewHandle
             bcc   GotHandle

;
;  If we can't get our DP, we return with the error in A and the carry set.
;
             tay                        ; save the error code
             pla
             pla                        ; get rid of the NIL handle
             pla                        ; pop our user ID
             plx                        ; pop the saved X register
             tya
             cmp #1
             rts                        ; and back out

GotHandle    tsc                        ; put the stack pointer
             tcd                        ; into the direct page
             lda   [1]                  ; the handle on DP
             sta   OurDP
             tcd                        ; and set our direct page from it
             pla
             pla                        ; we don't need the handle

             pla                        ; our User ID
             sta   <MyID                ; save it on DP

             plx                        ; the saved X register

             lda   #0
             sec                        ; we had to create the DP
             rts

GotDP        tcd
             lda   #0
             clc
             rts                        ; we already had the DP

             END

**********************************************************************
*
* CheckTheWorld looks at the output of MakeOurWorld and sees if
* something is really wrong.  If it is, it branches to EndOurWorld
* for us.  If not, it returns to the caller.
*
* This is for the use of those calls which don't need to know whether
* or not our direct page already existed.  If they do, they need to
* check individually and not rely on this routine.
*
*  Inputs:   Parameters to remove (in bytes) in Y
*            Takes Accumulator and carry flag values from MakeOurWorld
*
* Outputs:   Either returns or goes to EndOurWorld
*
**********************************************************************

CheckTheWorld  START

             bcc   WorldOK
             pha
             pla
             beq   WorldOK              ; only if carry is set

             tax                        ; put error code in X
             pla                        ; pop the RTS address
             bra   EndOurWorld          ; and get out of here

WorldOK      rts

             END

**********************************************************************
*
*  EndOurWorld isn't really pessimistic.  It restores the direct page
*  to the value MakeOurWorld pushed on the stack.  It removes the number
*  of bytes above the two RTL addresses specified in the Y register
*  and conditions the carry based on an error code passed in X.
*
*  Inputs:  Direct page on the stack, number of parameters to remove in
*           Y and an error code in X
*
* Outputs:  Removes Y parameters and RTLs with the carry conditioned and
*           the error code in the accumulator.
*
**********************************************************************

EndOurWorld  START

             cpx   #$FFFF               ; should we change the error?
             beq   NoErrorChange

             stx   <OurError            ; for PrError's future use
             dex

NoErrorChange inx                       ; This was Dave's idea.  Sheesh.

;
;  For $FFFF the dex is skipped, leaving X as zero
;

             plb                        ; reset the old DBR
             pld                        ; reset the old DP

             phy                        ; save the # of parameters to remove
             tsc                        ; take the stack with Y into A
             clc
             adc   1,s                  ; add the # of parameters to the SP
             tay                        ; and save as an index

             tsc                        ; another copy of the SP
             phx                        ; save the error code
             tax                        ; and index into the stack as well

             phb                        ; save the DBR so we can
             pea   0
             plb
             plb                        ; set it to zero for 16-bit addressing

;
;  With X equal to the stack pointer and Y equal to the stack pointer plus
;  the number of parameters to remove and B in bank zero, we can use addressing
;  like 3,x and 3,y to simulate stack addressing using X and Y as two different
;  stack pointers.  The toolbox does this and it's pretty nifty.
;
             lda   |7,x
             sta   |7,y                 ; moved two bytes
             lda   |5,x
             sta   |5,y                 ; moved four bytes
             lda   |3,x
             sta   |3,y                 ; moved all six bytes of RTL addresses

             plb                        ; restore the DBR
             plx                        ; restore the error code

             tsc                        ; yet another stack pointer copy
             clc
             adc   1,s                  ; add the # of parameters
             adc   #2                   ; and don't forget the pcount!
             tcs                        ; now the stack is moved

RealGone     txa
             cmp #1                     ; condition the carry
             rtl

             END

**********************************************************************
*
* OpenPICTFile increments the last letter of the supplied pathname
* until it finds one that doesn't exist or gets some other GS/OS error.
* It then opens the file and stores the ref num on direct page.  It
* then opens a QD II Picture for the current port and puts the picture
* handle on direct page.
*
*  Inputs:   RectPointer on DP for the Picture's frameRect
*
* Outputs:   PictHandle and FileRefNum on direct page
*
**********************************************************************

OpenPICTFile START
             USING PicterData

             stz   <PictHandle+2
             stz   <PictHandle
             stz   <RefNum

GFILoop      _GetFileInfoGS GFIParams   ; does it exist?
             bcc   FileExists
             cmp   #$0046               ; file not found?
             beq   GotFileName
             brl   ExitOpen             ; get out of this routine

FileExists   shortm                     ; 8-bit land
             inc   FileNameSuffix
             longm
             bra   GFILoop

GotFileName  _CreateGS CreateParams
             bcs   ExitOpen

             _OpenGS OpenParams
             bcs   ExitOpen
             lda   OpenRefNum
             sta   <RefNum

DoPicture    pha
             pha                        ; space for result

             pei   <RectPointer+2
             pei   <RectPointer
             _OpenPicture
             tax
             pulllong <PictHandle
             txa

ExitOpen     rts                        ; carry's still properly set

             END

**********************************************************************
*
* ClosePICTFile writes the data in RefNum on direct page to the
* file opened with RefNum, closes the file and kills the picture.
*
*  Inputs:   PictHandle and RefNum on direct page
*
* Outputs:   None
*
**********************************************************************

ClosePICTFile      START
             USING PicterData

             lda   <PictHandle+1
             beq   NoPict
             _ClosePicture              ; first things first

NoPict       lda   <RefNum
             beq   NoFile
             sta   WriteRefNum
             sta   CloseRefNum

             pha
             pha                        ; space for result
             pei   <PictHandle+2
             pei   <PictHandle
             _GetHandleSize
             pulllong WriteReqCount     ; how much to write

             ldy   #4
             lda   [<PictHandle],y
             pha                        ; save handle attribs and lock
             ora   #$8000
             sta   [<PictHandle],y
             dey
             dey
             lda   [<PictHandle],y
             sta   WriteBuffer+2
             lda   [<PictHandle]
             sta   WriteBuffer          ; pointer to the picture
             _WriteGS WriteParams

             tax
             pla
             ldy   #4
             sta   [<PictHandle],y
             phx

             _CloseGS CloseParams
             ply                        ; the error from Write, if any
             tax
             bcs   ExitClose

             tyx
             cpx   #1
             bcs   ExitClose            ; handle write errors, if any

NoFile       lda   <PictHandle+1
             beq   StillNoPict
             pei   <PictHandle+2
             pei   <PictHandle
             _KillPicture

StillNoPict  ldx   #0

ExitClose    txa
             cmp   #1
             rts

             END

**********************************************************************
*
* DoIdleProc calls the pIdleProc procedure if it is not nil.
*
*  Inputs:   PrRecPtr on direct page
*
* Outputs:   None
*
**********************************************************************

DoIdleProc   START

             ldy   #opIdleProc+2
             lda   [<PrRecPtr],y
             dey
             dey
             ora   [<PrRecPtr],y
             beq   ExitIdleLoop         ; nothing to call

             lda   [<PrRecPtr],y        ; low and middle bytes of pointer
             sta   IdleJSL+1
             iny
             lda   [<PrRecPtr],y
             sta   IdleJSL+2            ; middle and high bytes of pointer

IdleJSL      jsl   IdleJSL              ; self-modified (ack)

ExitIdleLoop rts

             END

*****************************************************************
*
*  Ben's dialog routines call a lot of subroutines to set and
*  retrieve items in the print record.  Rather than modify the
*  dialog routines to be more specific, we include these subroutines
*  in this segment.  Each of them assumes <PrRecPtr points to the
*  print record in question.  The "set" routines take the parameter
*  in the registers (as described for each routine) and the "get"
*  routines return parameters similarly.
*
*  These routines must be modified for your printer driver to return
*  the appopriate values.  These values are right for $8001 drivers,
*  which interpret the style subrecord as documented for the ImageWriter
*  driver in Toolbox Reference, Volume 1.
*
******************************************************************

Metrics      START
             USING PicterData

; GetPaper returns the paper type in A.

GetPaper     ENTRY
             ldy   #oprStl+opaperType
             lda   [<PrRecPtr],y
             rts

; SetPaper takes the paper type in OptionA (not like GetPaper)

SetPaper     ENTRY
             ldy   #oprStl+opaperType
             lda   <OptionA
             sta   [<PrRecPtr],y
             rts

; GetReduction returns the reduction (100/50 = 0/1) in A.

GetReduction ENTRY
             ldy   #oprStl+owDev
             lda   [<PrRecPtr],y
             and   #%0000000000001000   ; reduction bit in wDev
             bne   ReductionIs100       ; in the print record, the bit clear
Reductionis50 lda #1                    ; means 50% reduction, which is reverse
             bra   GetReducOut          ; of how we want it
Reductionis100 lda #0
GetReducOut  rts

; SetReduction takes a value in OptionA and sets page reduction to 50% if
; OptionA is 1 and to 100% if OptionA is 0.

SetReduction ENTRY
             lda   <OptionA
             asl   a
             asl   a
             asl   a                    ; the bit is now in bit 3
             eor   #wDevReduction       ; flip the bit
             ldy   #oprStl+owDev
             pha
             lda   [<PrRecPtr],y
             and   #$FFF7               ; clear out our bit
             ora   1,s                  ; set it appopriately
             sta   [<PrRecPtr],y
             pla
             jsr   CalcMetrics          ; fix up the paper sizes
             rts

; GetPageGap gets page gap (yes/no) in the zero flag:  BEQ succeds if there
; is a normal page gap and BNE succeeds if not

GetPageGap   ENTRY
             ldy   #oprStl+owDev
             lda   [<PrRecPtr],y
             and   #wDevGap
             eor   #wDevGap             ; flip the bit
             rts

; SetPageGap takes a word in OptionA - zero for regular gap and non-zero
; for no gap

SetPageGap   ENTRY
             ldy   #oprStl+owDev
             lda   [<PrRecPtr],y
             ldx   <OptionA             ; the value to use
             bne   SetNoGap             ; non-zero = no gap
SetGap       ora   #wDevGap             ; set the bit
             bra   StoreGap
SetNoGap     and   #$FFFF-wDevGap       ; set the bit
StoreGap     sta   [<PrRecPtr],y
             jsr   CalcMetrics          ; fix things up
             rts

; GetVSize returns vertical size in the zero flag (BEQ taken if normal size)

GetVSize     ENTRY

             ldy   #oprStl+owDev
             lda   [<PrRecPtr],y
             and   #wDevVSize
             eor   #wDevVSize
             rts

; SetVSize takes 0 for normal and non-zero for condensed in OptionA and sets
; the print record accordingly

SetVSize     ENTRY
             ldy   #oprStl+owDev
             lda   [<PrRecPtr],y
             ldx   <OptionA             ; the value to use
             bne   SetCondensed         ; non-zero = no gap
SetNormal    ora   #wDevVSize           ; set the bit
             bra   StoreVSize
SetCondensed and   #$FFFF-wDevVSize     ; set the bit
StoreVSize   sta   [<PrRecPtr],y
             jsr   CalcMetrics          ; fix things up
             rts

; GetOrientation returns 1 in A for landscape, zero for portrait

GetOrientation ENTRY
             ldy   #oprStl+owDev
             lda   [<PrRecPtr],y
             and   #wDevOrientation
             eor   #wDevOrientation
             beq   GetOrientOut
             lda   #1
GetOrientOut rts

; SetOrientation takes 0 in OptionA for portrait or non-zero for landscape

SetOrientation ENTRY

             ldy   #oprStl+owDev
             lda   [<PrRecPtr],y
             ldx   <OptionA             ; the value to use
             bne   SetLand              ; non-zero = landscape
SetPortrait  ora   #wDevOrientation     ; set the bit
             bra   StoreOrient
SetLand      and   #$FFFF-wDevOrientation ; set the bit
StoreOrient  sta   [<PrRecPtr],y
             jsr   CalcMetrics          ; fix things up
             rts

; GetQuality returns the quality in A: 0 for standard, 1 for better

GetQuality   ENTRY

             ldy   #oprStl+owDev
             lda   [<PrRecPtr],y
             and   #wDevBetterText
             beq   GetQualOut           ; if zero, we're done
             lda   #1
GetQualOut   rts

; SetQuality takes 0 for standard or 1 for better in OptionA and sets
; the print record accordingly

SetQuality   ENTRY
             ldy   #oPrStl+owDev
             lda   [<PrRecPtr],y
             and   #$FFFF-wDevBetterText ; clear out bit zero
             ora   <OptionA             ; set it if necessary
             sta   [<PrRecPtr],y

;
; This driver doesn't support both draft and spool printing, but if it did,
; or if you do, this is where to set the parameter in the job subrecord.  If
; you know it's draft printing, assuming you use zero for draft printing,
; then you can do this:
;
;            shortm
;            lda   #0                   ; draft printing
;            ldy   #oprJob+oBJDocLoop   ; the BJDocLoop byte
;            sta   [<PrRecPtr],y
;            longm
;            stz   <OptionA
;            jsr   SetReduction         ; zero in OptionA = 100%
;            jsr   SetPageGap           ; zero in OptionA = default gap
;            jsr   SetOrientation       ; zero in OptioNA = portrait
;

             rts

; GetColor returns the color in the zero flag = BEQ if no color, BNE if color

GetColor     ENTRY
             ldy   #oprStl+owDev
             lda   [<PrRecPtr],y
             and   #wDevColor
             rts

; SetColor takes zero in OptionA for B/W and non-zero for color

SetColor     ENTRY
             ldy   #oprStl+owDev
             lda   [<PrRecPtr],y
             ldx   <OptionA
             beq   SetBW                ; black and white
SetRainbow   ora   #wDevColor
             bra   StoreColor
SetBW        and   #$FFFF-wDevColor
StoreColor   sta   [<PrRecPtr],y
             rts

; GetFeed gets manual page option in zero flag (BEQ works if manual feed)

GetFeed      ENTRY
             ldy   #oprStl+oFeed
             lda   [<PrRecPtr],y
             rts

; SetFeed sets the manual page option (0 = manual feed)

SetFeed      ENTRY
             ldy   #oprStl+oFeed
             lda   <OptionA
             sta   [<PrRecPtr],y
             jsr   CalcMetrics
             rts

; SetPageRange takes the first page in OptionA and the last page in OptionB

SetPageRange ENTRY
             ldy   #oprJob+oiFstPage
             lda   <OptionA
             sta   [<PrRecPtr],y
             ldy   #oprJob+oiLstPage
             lda   <OptionB
             sta   [<PrRecPtr],y
             jsr   CalcMetrics
             rts

; SetCopies sets the number of copies from OptionA

SetCopies    ENTRY
             ldy   #oprJob+oiCopies
             lda   <OptionA
             sta   [<PrRecPtr],y
             jsr   CalcMetrics
             rts

; CalcMetrics makes sure the resolutions, width, rPage and rPaper are right
; for the values in the print record

CalcMetrics  ENTRY

; we don't use landscape or lots of paper types - all we do is look at the
; paper type and make sure the page and paper rectangles are correct

;
;  We have four possibilities for rectangles - screen 320, screen 640 and
;  US letter 320 and US letter 640.  First we check to be sure that the
;  paper type is either US letter or screen sized, and then blast in the
;  appropriate page and paper rectangles (easier than comparing them)
;

             ldy   #oprStl+opaperType
             lda   [PrRecPtr],y
             cmp   #paperTypeUSALetter
             bne   KeepChecking
             brl   RectLetter
KeepChecking cmp   #4
             bne   BadRects
             brl   RectScreen
BadRects     lda   #$FFFF               ; flag to do a default
             sta   <RectChanged
             brl   DoneRects            ; get out of here

RectLetter   pha
             _GetMasterSCB
             pla
             sta   <MasterSCB
             and   #$0080
             beq   RectLetter320

RectLetter640 ldy  #oprInfo+orPage
RectLoop1    lda   [PrRecPtr],y
             cmp   FPrPage640-orPage-oprInfo,y
             beq   OK1

             pushlong #FPrPage640
             lda   <PrRecPtr
             clc
             adc   #oprInfo+orPage
             tax
             lda   <PrRecPtr+2
             adc   #0
             pha
             phx                        ; pointer to destination
             pushlong #16               ; two rects = 16 bytes
             _BlockMove
             inc   <RectChanged
             brl   DoneRects

OK1          iny
             iny
             cpy   #oprInfo+orPage+16
             bne   RectLoop1
             brl   DoneRects

RectLetter320 ldy #oprInfo+orPage
RectLoop2    lda   [PrRecPtr],y
             cmp   FPrPage320-orPage-oprInfo,y
             beq   OK2

             pushlong #FPrPage320
             lda   <PrRecPtr
             clc
             adc   #oprInfo+orPage
             tax
             lda   <PrRecPtr+2
             adc   #0
             pha
             phx                        ; pointer to destination
             pushlong #16               ; two rects = 16 bytes
             _BlockMove
             inc   <RectChanged
             brl   DoneRects

OK2          iny
             iny
             cpy   #oprInfo+orPage+16
             bne   RectLoop2
             brl   DoneRects

RectScreen   lda   <MasterSCB
             and   #$0080
             beq   RectScreen320

RectScreen640 ldy  #oprInfo+orPage
RectLoop3    lda   [PrRecPtr],y
             cmp   SCrPage640-orPage-oprInfo,y
             beq   OK3

             pushlong #SCrPage640
             lda   <PrRecPtr
             clc
             adc   #oprInfo+orPage
             tax
             lda   <PrRecPtr+2
             adc   #0
             pha
             phx                        ; pointer to destination
             pushlong #16               ; two rects = 16 bytes
             _BlockMove
             inc   <RectChanged
             brl   DoneRects

OK3          iny
             iny
             cpy   #oprInfo+orPage+16
             bne   RectLoop3
             brl   DoneRects

RectScreen320 ldy  #oprInfo+orPage
RectLoop4    lda   [PrRecPtr],y
             cmp   SCrPage320-orPage-oprInfo,y
             beq   OK4

             pushlong #SCrPage320
             lda   <PrRecPtr
             clc
             adc   #oprInfo+orPage
             tax
             lda   <PrRecPtr+2
             adc   #0
             pha
             phx                        ; pointer to destination
             pushlong #16               ; two rects = 16 bytes
             _BlockMove
             inc   <RectChanged
             bra   DoneRects

OK4          iny
             iny
             cpy   #oprInfo+orPage+16
             beq   doneRects
             bra   RectLoop4

DoneRects    rts

             END
